// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2018 MediaTek Inc.
 *
 * Author: ZH Chen <zh.chen@mediatek.com>
 *
 */

#include "pinctrl-mtk-mt6763.h"
#include "pinctrl-paris.h"

/* MT6763 have multiple bases to program pin configuration listed as the below:
 * gpio:0x10005000,      io_cfg_lt:0x11F20000, io_cfg_lm:0x11E80000,
 * io_cfg_lb:0x11E70000, io_cfg_bl:0x11E90000, io_cfg_rb:0x11D30000,
 * io_cfg_rm:0x11D20000, io_cfg_rt:0x11C50000, io_cfg_tl:0x11F10000
 * _i_based could be used to indicate what base the pin should be mapped into.
 */

#define PIN_FIELD_BASE(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits) \
	PIN_FIELD_CALC(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits, \
		       32, 0)

#define PINS_FIELD_BASE(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits) \
	PIN_FIELD_CALC(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits,  \
		       32, 1)

static const struct mtk_pin_field_calc mt6763_pin_mode_range[] = {
	PIN_FIELD_BASE(0, 7, 0, 0x0300, 0x10, 0, 4),
	PIN_FIELD_BASE(8, 15, 0, 0x0310, 0x10, 0, 4),
	PIN_FIELD_BASE(16, 23, 0, 0x0320, 0x10, 0, 4),
	PIN_FIELD_BASE(24, 31, 0, 0x0330, 0x10, 0, 4),
	PIN_FIELD_BASE(32, 39, 0, 0x0340, 0x10, 0, 4),
	PIN_FIELD_BASE(40, 47, 0, 0x0350, 0x10, 0, 4),
	PIN_FIELD_BASE(48, 55, 0, 0x0360, 0x10, 0, 4),
	PIN_FIELD_BASE(56, 63, 0, 0x0370, 0x10, 0, 4),
	PIN_FIELD_BASE(64, 71, 0, 0x0380, 0x10, 0, 4),
	PIN_FIELD_BASE(72, 79, 0, 0x0390, 0x10, 0, 4),
	PIN_FIELD_BASE(80, 87, 0, 0x03A0, 0x10, 0, 4),
	PIN_FIELD_BASE(88, 95, 0, 0x03B0, 0x10, 0, 4),
	PIN_FIELD_BASE(96, 103, 0, 0x03C0, 0x10, 0, 4),
	PIN_FIELD_BASE(104, 111, 0, 0x03D0, 0x10, 0, 4),
	PIN_FIELD_BASE(112, 119, 0, 0x03E0, 0x10, 0, 4),
	PIN_FIELD_BASE(120, 127, 0, 0x03F0, 0x10, 0, 4),
	PIN_FIELD_BASE(128, 135, 0, 0x0400, 0x10, 0, 4),
	PIN_FIELD_BASE(136, 143, 0, 0x0410, 0x10, 0, 4),
	PIN_FIELD_BASE(144, 151, 0, 0x0420, 0x10, 0, 4),
};

static const struct mtk_pin_field_calc mt6763_pin_dir_range[] = {
	PIN_FIELD_BASE(0, 31, 0, 0x0000, 0x10, 0, 1),
	PIN_FIELD_BASE(32, 63, 0, 0x0010, 0x10, 0, 1),
	PIN_FIELD_BASE(64, 95, 0, 0x0020, 0x10, 0, 1),
	PIN_FIELD_BASE(96, 127, 0, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(128, 151, 0, 0x0040, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_di_range[] = {
	PIN_FIELD_BASE(0, 31, 0, 0x200, 0x10, 0, 1),
	PIN_FIELD_BASE(32, 63, 0, 0x210, 0x10, 0, 1),
	PIN_FIELD_BASE(64, 95, 0, 0x220, 0x10, 0, 1),
	PIN_FIELD_BASE(96, 127, 0, 0x230, 0x10, 0, 1),
	PIN_FIELD_BASE(128, 151, 0, 0x240, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_do_range[] = {
	PIN_FIELD_BASE(0, 31, 0, 0x0100, 0x10, 0, 1),
	PIN_FIELD_BASE(32, 63, 0, 0x0110, 0x10, 0, 1),
	PIN_FIELD_BASE(64, 95, 0, 0x0120, 0x10, 0, 1),
	PIN_FIELD_BASE(96, 127, 0, 0x0130, 0x10, 0, 1),
	PIN_FIELD_BASE(128, 151, 0, 0x0140, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_ies_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(1, 1, 6, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(2, 2, 6, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(3, 3, 6, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(4, 4, 6, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(5, 5, 6, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(6, 6, 6, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(7, 7, 6, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(8, 8, 6, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(9, 9, 6, 0x000, 0x10, 11, 1),
	PIN_FIELD_BASE(10, 10, 6, 0x000, 0x10, 11, 1),
	PIN_FIELD_BASE(11, 11, 1, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(12, 12, 1, 0x000, 0x10, 7, 1),
	PIN_FIELD_BASE(13, 13, 2, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(14, 14, 2, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(15, 15, 2, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(16, 16, 2, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(17, 17, 2, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(18, 18, 2, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(19, 19, 2, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(20, 20, 2, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(21, 21, 2, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(22, 22, 2, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(23, 23, 2, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(24, 24, 2, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(25, 25, 2, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(26, 26, 2, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(27, 27, 2, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(28, 28, 2, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(29, 29, 2, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(30, 30, 2, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(31, 31, 2, 0x000, 0x10, 6, 1),
	PIN_FIELD_BASE(32, 32, 2, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(33, 33, 2, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(34, 34, 2, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(35, 35, 3, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(36, 36, 3, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(37, 37, 3, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(38, 38, 3, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(39, 39, 3, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(40, 40, 3, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(41, 41, 3, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(42, 42, 3, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(43, 43, 3, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(44, 44, 3, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(45, 45, 3, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(46, 46, 3, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(47, 47, 3, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(48, 48, 3, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(49, 49, 3, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(50, 50, 4, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(51, 51, 4, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(52, 52, 4, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(53, 53, 4, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(54, 54, 4, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(55, 55, 4, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(56, 56, 4, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(57, 57, 4, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(58, 58, 4, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(59, 59, 4, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(60, 60, 4, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(61, 61, 5, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(62, 62, 5, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(63, 63, 5, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(64, 64, 5, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(65, 65, 5, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(66, 66, 5, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(67, 67, 5, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(68, 68, 5, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(69, 69, 5, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(70, 70, 5, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(71, 71, 5, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(72, 72, 5, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(73, 73, 5, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(74, 74, 5, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(75, 75, 5, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(76, 76, 5, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(77, 77, 5, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(78, 78, 5, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(79, 79, 5, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(80, 80, 5, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(81, 81, 5, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(82, 82, 5, 0x000, 0x10, 6, 1),
	PIN_FIELD_BASE(83, 83, 5, 0x000, 0x10, 6, 1),
	PIN_FIELD_BASE(84, 84, 5, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(85, 85, 5, 0x000, 0x10, 7, 1),
	PIN_FIELD_BASE(86, 86, 5, 0x000, 0x10, 7, 1),
	PIN_FIELD_BASE(87, 87, 5, 0x000, 0x10, 7, 1),
	PIN_FIELD_BASE(88, 88, 5, 0x000, 0x10, 7, 1),
	PIN_FIELD_BASE(89, 89, 6, 0x000, 0x10, 10, 1),
	PIN_FIELD_BASE(90, 90, 6, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(91, 91, 6, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(92, 92, 6, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(93, 93, 6, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(94, 94, 6, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(95, 95, 6, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(96, 96, 6, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(97, 97, 6, 0x000, 0x10, 6, 1),
	PIN_FIELD_BASE(98, 98, 6, 0x000, 0x10, 6, 1),
	PIN_FIELD_BASE(99, 99, 6, 0x000, 0x10, 7, 1),
	PIN_FIELD_BASE(100, 100, 6, 0x000, 0x10, 8, 1),
	PIN_FIELD_BASE(101, 101, 6, 0x000, 0x10, 9, 1),
	PIN_FIELD_BASE(102, 102, 6, 0x000, 0x10, 9, 1),
	PIN_FIELD_BASE(103, 103, 6, 0x000, 0x10, 12, 1),
	PIN_FIELD_BASE(104, 104, 6, 0x000, 0x10, 12, 1),
	PIN_FIELD_BASE(105, 105, 6, 0x000, 0x10, 13, 1),
	PIN_FIELD_BASE(106, 106, 6, 0x000, 0x10, 13, 1),
	PIN_FIELD_BASE(107, 107, 7, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(108, 108, 7, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(109, 109, 7, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(110, 110, 7, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(111, 111, 7, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(112, 112, 7, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(113, 113, 7, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(114, 114, 7, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(115, 115, 7, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(116, 116, 7, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(117, 117, 7, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(118, 118, 7, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(119, 119, 7, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(120, 120, 7, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(121, 121, 7, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(122, 122, 8, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(123, 123, 8, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(124, 124, 8, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(125, 125, 8, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(126, 126, 8, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(127, 127, 8, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(128, 128, 8, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(129, 129, 8, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(130, 130, 8, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(131, 131, 8, 0x000, 0x10, 3, 1),
	PIN_FIELD_BASE(132, 132, 8, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(133, 133, 8, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(134, 134, 1, 0x000, 0x10, 0, 1),
	PIN_FIELD_BASE(135, 135, 1, 0x000, 0x10, 1, 1),
	PIN_FIELD_BASE(136, 136, 1, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(137, 137, 1, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(138, 138, 1, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(139, 139, 1, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(140, 140, 1, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(141, 141, 1, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(142, 142, 1, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(143, 143, 1, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(144, 144, 1, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(145, 145, 1, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(146, 146, 1, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(147, 147, 1, 0x000, 0x10, 4, 1),
	PIN_FIELD_BASE(148, 148, 1, 0x000, 0x10, 5, 1),
	PIN_FIELD_BASE(149, 149, 1, 0x000, 0x10, 6, 1),
	PIN_FIELD_BASE(150, 150, 4, 0x000, 0x10, 2, 1),
	PIN_FIELD_BASE(151, 151, 4, 0x000, 0x10, 2, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_smt_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(1, 1, 6, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(2, 2, 6, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(3, 3, 6, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(4, 4, 6, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(5, 5, 6, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(6, 6, 6, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(7, 7, 6, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(8, 8, 6, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(9, 9, 6, 0x010, 0x10, 11, 1),
	PIN_FIELD_BASE(10, 10, 6, 0x010, 0x10, 11, 1),
	PIN_FIELD_BASE(11, 11, 1, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(12, 12, 1, 0x010, 0x10, 7, 1),
	PIN_FIELD_BASE(13, 13, 2, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(14, 14, 2, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(15, 15, 2, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(16, 16, 2, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(17, 17, 2, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(18, 18, 2, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(19, 19, 2, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(20, 20, 2, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(21, 21, 2, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(22, 22, 2, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(23, 23, 2, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(24, 24, 2, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(25, 25, 2, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(26, 26, 2, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(27, 27, 2, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(28, 28, 2, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(29, 29, 2, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(30, 30, 2, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(31, 31, 2, 0x010, 0x10, 6, 1),
	PIN_FIELD_BASE(32, 32, 2, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(33, 33, 2, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(34, 34, 2, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(35, 35, 3, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(36, 36, 3, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(37, 37, 3, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(38, 38, 3, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(39, 39, 3, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(40, 40, 3, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(41, 41, 3, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(42, 42, 3, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(43, 43, 3, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(44, 44, 3, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(45, 45, 3, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(46, 46, 3, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(47, 47, 3, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(48, 48, 3, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(49, 49, 3, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(50, 50, 4, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(51, 51, 4, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(52, 52, 4, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(53, 53, 4, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(54, 54, 4, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(55, 55, 4, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(56, 56, 4, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(57, 57, 4, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(58, 58, 4, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(59, 59, 4, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(60, 60, 4, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(61, 61, 5, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(62, 62, 5, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(63, 63, 5, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(64, 64, 5, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(65, 65, 5, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(66, 66, 5, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(67, 67, 5, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(68, 68, 5, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(69, 69, 5, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(70, 70, 5, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(71, 71, 5, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(72, 72, 5, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(73, 73, 5, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(74, 74, 5, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(75, 75, 5, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(76, 76, 5, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(77, 77, 5, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(78, 78, 5, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(79, 79, 5, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(80, 80, 5, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(81, 81, 5, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(82, 82, 5, 0x010, 0x10, 6, 1),
	PIN_FIELD_BASE(83, 83, 5, 0x010, 0x10, 6, 1),
	PIN_FIELD_BASE(84, 84, 5, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(85, 85, 5, 0x010, 0x10, 7, 1),
	PIN_FIELD_BASE(86, 86, 5, 0x010, 0x10, 7, 1),
	PIN_FIELD_BASE(87, 87, 5, 0x010, 0x10, 7, 1),
	PIN_FIELD_BASE(88, 88, 5, 0x010, 0x10, 7, 1),
	PIN_FIELD_BASE(89, 89, 6, 0x010, 0x10, 10, 1),
	PIN_FIELD_BASE(90, 90, 6, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(91, 91, 6, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(92, 92, 6, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(93, 93, 6, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(94, 94, 6, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(95, 95, 6, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(96, 96, 6, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(97, 97, 6, 0x010, 0x10, 6, 1),
	PIN_FIELD_BASE(98, 98, 6, 0x010, 0x10, 6, 1),
	PIN_FIELD_BASE(99, 99, 6, 0x010, 0x10, 7, 1),
	PIN_FIELD_BASE(100, 100, 6, 0x010, 0x10, 8, 1),
	PIN_FIELD_BASE(101, 101, 6, 0x010, 0x10, 9, 1),
	PIN_FIELD_BASE(102, 102, 6, 0x010, 0x10, 9, 1),
	PIN_FIELD_BASE(103, 103, 6, 0x010, 0x10, 12, 1),
	PIN_FIELD_BASE(104, 104, 6, 0x010, 0x10, 12, 1),
	PIN_FIELD_BASE(105, 105, 6, 0x010, 0x10, 13, 1),
	PIN_FIELD_BASE(106, 106, 6, 0x010, 0x10, 13, 1),
	PIN_FIELD_BASE(107, 107, 7, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(108, 108, 7, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(109, 109, 7, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(110, 110, 7, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(111, 111, 7, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(112, 112, 7, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(113, 113, 7, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(114, 114, 7, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(115, 115, 7, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(116, 116, 7, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(117, 117, 7, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(118, 118, 7, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(119, 119, 7, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(120, 120, 7, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(121, 121, 7, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(122, 122, 8, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(123, 123, 8, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(124, 124, 8, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(125, 125, 8, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(126, 126, 8, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(127, 127, 8, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(128, 128, 8, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(129, 129, 8, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(130, 130, 8, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(131, 131, 8, 0x010, 0x10, 3, 1),
	PIN_FIELD_BASE(132, 132, 8, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(133, 133, 8, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(134, 134, 1, 0x010, 0x10, 0, 1),
	PIN_FIELD_BASE(135, 135, 1, 0x010, 0x10, 1, 1),
	PIN_FIELD_BASE(136, 136, 1, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(137, 137, 1, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(138, 138, 1, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(139, 139, 1, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(140, 140, 1, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(141, 141, 1, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(142, 142, 1, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(143, 143, 1, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(144, 144, 1, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(145, 145, 1, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(146, 146, 1, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(147, 147, 1, 0x010, 0x10, 4, 1),
	PIN_FIELD_BASE(148, 148, 1, 0x010, 0x10, 5, 1),
	PIN_FIELD_BASE(149, 149, 1, 0x010, 0x10, 6, 1),
	PIN_FIELD_BASE(150, 150, 4, 0x010, 0x10, 2, 1),
	PIN_FIELD_BASE(151, 151, 4, 0x010, 0x10, 2, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_pullen_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x060, 0x10, 6, 1),
	PIN_FIELD_BASE(1, 1, 6, 0x060, 0x10, 7, 1),
	PIN_FIELD_BASE(2, 2, 6, 0x060, 0x10, 8, 1),
	PIN_FIELD_BASE(3, 3, 6, 0x060, 0x10, 9, 1),
	PIN_FIELD_BASE(4, 4, 6, 0x060, 0x10, 10, 1),
	PIN_FIELD_BASE(5, 5, 6, 0x060, 0x10, 11, 1),
	PIN_FIELD_BASE(6, 6, 6, 0x060, 0x10, 12, 1),
	PIN_FIELD_BASE(7, 7, 6, 0x060, 0x10, 13, 1),
	PIN_FIELD_BASE(8, 8, 6, 0x060, 0x10, 22, 1),
	PIN_FIELD_BASE(9, 9, 6, 0x060, 0x10, 23, 1),
	PIN_FIELD_BASE(10, 10, 6, 0x060, 0x10, 24, 1),
	PIN_FIELD_BASE(11, 11, 1, 0x060, 0x10, 10, 1),
	PIN_FIELD_BASE(12, 12, 1, 0x060, 0x10, 17, 1),
	PIN_FIELD_BASE(13, 13, 2, 0x060, 0x10, 0, 1),
	PIN_FIELD_BASE(14, 14, 2, 0x060, 0x10, 1, 1),
	PIN_FIELD_BASE(15, 15, 2, 0x060, 0x10, 2, 1),
	PIN_FIELD_BASE(16, 16, 2, 0x060, 0x10, 3, 1),
	PIN_FIELD_BASE(17, 17, 2, 0x060, 0x10, 4, 1),
	PIN_FIELD_BASE(18, 18, 2, 0x060, 0x10, 5, 1),
	PIN_FIELD_BASE(19, 19, 2, 0x060, 0x10, 6, 1),
	PIN_FIELD_BASE(20, 20, 2, 0x060, 0x10, 7, 1),
	PIN_FIELD_BASE(21, 21, 2, 0x060, 0x10, 8, 1),
	PIN_FIELD_BASE(22, 22, 2, 0x060, 0x10, 9, 1),
	PIN_FIELD_BASE(23, 23, 2, 0x060, 0x10, 10, 1),
	PIN_FIELD_BASE(24, 24, 2, 0x060, 0x10, 11, 1),
	PIN_FIELD_BASE(25, 25, 2, 0x060, 0x10, 12, 1),
	PIN_FIELD_BASE(26, 26, 2, 0x060, 0x10, 13, 1),
	PIN_FIELD_BASE(27, 27, 2, 0x060, 0x10, 14, 1),
	PIN_FIELD_BASE(28, 28, 2, 0x060, 0x10, 15, 1),
	PIN_FIELD_BASE(43, 43, 3, 0x060, 0x10, 8, 1),
	PIN_FIELD_BASE(44, 44, 3, 0x060, 0x10, 9, 1),
	PIN_FIELD_BASE(45, 45, 3, 0x060, 0x10, 10, 1),
	PIN_FIELD_BASE(46, 46, 3, 0x060, 0x10, 11, 1),
	PIN_FIELD_BASE(47, 47, 3, 0x060, 0x10, 12, 1),
	PIN_FIELD_BASE(48, 48, 3, 0x060, 0x10, 13, 1),
	PIN_FIELD_BASE(49, 49, 3, 0x060, 0x10, 14, 1),
	PIN_FIELD_BASE(50, 50, 4, 0x060, 0x10, 0, 1),
	PIN_FIELD_BASE(51, 51, 4, 0x060, 0x10, 1, 1),
	PIN_FIELD_BASE(52, 52, 4, 0x060, 0x10, 2, 1),
	PIN_FIELD_BASE(53, 53, 4, 0x060, 0x10, 3, 1),
	PIN_FIELD_BASE(54, 54, 4, 0x060, 0x10, 4, 1),
	PIN_FIELD_BASE(55, 55, 4, 0x060, 0x10, 5, 1),
	PIN_FIELD_BASE(56, 56, 4, 0x060, 0x10, 6, 1),
	PIN_FIELD_BASE(57, 57, 4, 0x060, 0x10, 7, 1),
	PIN_FIELD_BASE(58, 58, 4, 0x060, 0x10, 8, 1),
	PIN_FIELD_BASE(59, 59, 4, 0x060, 0x10, 9, 1),
	PIN_FIELD_BASE(60, 60, 4, 0x060, 0x10, 10, 1),
	PIN_FIELD_BASE(61, 61, 5, 0x060, 0x10, 0, 1),
	PIN_FIELD_BASE(62, 62, 5, 0x060, 0x10, 1, 1),
	PIN_FIELD_BASE(63, 63, 5, 0x060, 0x10, 2, 1),
	PIN_FIELD_BASE(64, 64, 5, 0x060, 0x10, 3, 1),
	PIN_FIELD_BASE(65, 65, 5, 0x060, 0x10, 4, 1),
	PIN_FIELD_BASE(66, 66, 5, 0x060, 0x10, 5, 1),
	PIN_FIELD_BASE(67, 67, 5, 0x060, 0x10, 6, 1),
	PIN_FIELD_BASE(68, 68, 5, 0x060, 0x10, 7, 1),
	PIN_FIELD_BASE(69, 69, 5, 0x060, 0x10, 8, 1),
	PIN_FIELD_BASE(70, 70, 5, 0x060, 0x10, 9, 1),
	PIN_FIELD_BASE(71, 71, 5, 0x060, 0x10, 10, 1),
	PIN_FIELD_BASE(72, 72, 5, 0x060, 0x10, 11, 1),
	PIN_FIELD_BASE(73, 73, 5, 0x060, 0x10, 12, 1),
	PIN_FIELD_BASE(74, 74, 5, 0x060, 0x10, 13, 1),
	PIN_FIELD_BASE(75, 75, 5, 0x060, 0x10, 14, 1),
	PIN_FIELD_BASE(76, 76, 5, 0x060, 0x10, 15, 1),
	PIN_FIELD_BASE(77, 77, 5, 0x060, 0x10, 16, 1),
	PIN_FIELD_BASE(78, 78, 5, 0x060, 0x10, 17, 1),
	PIN_FIELD_BASE(79, 79, 5, 0x060, 0x10, 18, 1),
	PIN_FIELD_BASE(80, 80, 5, 0x060, 0x10, 19, 1),
	PIN_FIELD_BASE(81, 81, 5, 0x060, 0x10, 20, 1),
	PIN_FIELD_BASE(82, 82, 5, 0x060, 0x10, 21, 1),
	PIN_FIELD_BASE(83, 83, 5, 0x060, 0x10, 22, 1),
	PIN_FIELD_BASE(84, 84, 5, 0x060, 0x10, 23, 1),
	PIN_FIELD_BASE(85, 85, 5, 0x060, 0x10, 24, 1),
	PIN_FIELD_BASE(86, 86, 5, 0x060, 0x10, 25, 1),
	PIN_FIELD_BASE(87, 87, 5, 0x060, 0x10, 26, 1),
	PIN_FIELD_BASE(88, 88, 5, 0x060, 0x10, 27, 1),
	PIN_FIELD_BASE(89, 89, 6, 0x060, 0x10, 0, 1),
	PIN_FIELD_BASE(90, 90, 6, 0x060, 0x10, 1, 1),
	PIN_FIELD_BASE(95, 95, 6, 0x060, 0x10, 14, 1),
	PIN_FIELD_BASE(96, 96, 6, 0x060, 0x10, 15, 1),
	PIN_FIELD_BASE(97, 97, 6, 0x060, 0x10, 16, 1),
	PIN_FIELD_BASE(98, 98, 6, 0x060, 0x10, 17, 1),
	PIN_FIELD_BASE(99, 99, 6, 0x060, 0x10, 18, 1),
	PIN_FIELD_BASE(100, 100, 6, 0x060, 0x10, 19, 1),
	PIN_FIELD_BASE(101, 101, 6, 0x060, 0x10, 20, 1),
	PIN_FIELD_BASE(102, 102, 6, 0x060, 0x10, 21, 1),
	PIN_FIELD_BASE(103, 103, 6, 0x060, 0x10, 25, 1),
	PIN_FIELD_BASE(104, 104, 6, 0x060, 0x10, 26, 1),
	PIN_FIELD_BASE(105, 105, 6, 0x060, 0x10, 27, 1),
	PIN_FIELD_BASE(106, 106, 6, 0x060, 0x10, 28, 1),
	PIN_FIELD_BASE(110, 110, 7, 0x060, 0x10, 3, 1),
	PIN_FIELD_BASE(111, 111, 7, 0x060, 0x10, 4, 1),
	PIN_FIELD_BASE(112, 112, 7, 0x060, 0x10, 5, 1),
	PIN_FIELD_BASE(113, 113, 7, 0x060, 0x10, 6, 1),
	PIN_FIELD_BASE(114, 114, 7, 0x060, 0x10, 7, 1),
	PIN_FIELD_BASE(115, 115, 7, 0x060, 0x10, 8, 1),
	PIN_FIELD_BASE(134, 134, 1, 0x060, 0x10, 0, 1),
	PIN_FIELD_BASE(135, 135, 1, 0x060, 0x10, 1, 1),
	PIN_FIELD_BASE(136, 136, 1, 0x060, 0x10, 2, 1),
	PIN_FIELD_BASE(137, 137, 1, 0x060, 0x10, 3, 1),
	PIN_FIELD_BASE(138, 138, 1, 0x060, 0x10, 4, 1),
	PIN_FIELD_BASE(139, 139, 1, 0x060, 0x10, 5, 1),
	PIN_FIELD_BASE(140, 140, 1, 0x060, 0x10, 6, 1),
	PIN_FIELD_BASE(141, 141, 1, 0x060, 0x10, 7, 1),
	PIN_FIELD_BASE(142, 142, 1, 0x060, 0x10, 8, 1),
	PIN_FIELD_BASE(143, 143, 1, 0x060, 0x10, 9, 1),
	PIN_FIELD_BASE(144, 144, 1, 0x060, 0x10, 11, 1),
	PIN_FIELD_BASE(145, 145, 1, 0x060, 0x10, 12, 1),
	PIN_FIELD_BASE(146, 146, 1, 0x060, 0x10, 13, 1),
	PIN_FIELD_BASE(147, 147, 1, 0x060, 0x10, 14, 1),
	PIN_FIELD_BASE(148, 148, 1, 0x060, 0x10, 15, 1),
	PIN_FIELD_BASE(149, 149, 1, 0x060, 0x10, 16, 1),
	PIN_FIELD_BASE(150, 150, 4, 0x060, 0x10, 11, 1),
	PIN_FIELD_BASE(151, 151, 4, 0x060, 0x10, 12, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_pullsel_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x080, 0x10, 6, 1),
	PIN_FIELD_BASE(1, 1, 6, 0x080, 0x10, 7, 1),
	PIN_FIELD_BASE(2, 2, 6, 0x080, 0x10, 8, 1),
	PIN_FIELD_BASE(3, 3, 6, 0x080, 0x10, 9, 1),
	PIN_FIELD_BASE(4, 4, 6, 0x080, 0x10, 10, 1),
	PIN_FIELD_BASE(5, 5, 6, 0x080, 0x10, 11, 1),
	PIN_FIELD_BASE(6, 6, 6, 0x080, 0x10, 12, 1),
	PIN_FIELD_BASE(7, 7, 6, 0x080, 0x10, 13, 1),
	PIN_FIELD_BASE(8, 8, 6, 0x080, 0x10, 22, 1),
	PIN_FIELD_BASE(9, 9, 6, 0x080, 0x10, 23, 1),
	PIN_FIELD_BASE(10, 10, 6, 0x080, 0x10, 24, 1),
	PIN_FIELD_BASE(11, 11, 1, 0x080, 0x10, 10, 1),
	PIN_FIELD_BASE(12, 12, 1, 0x080, 0x10, 17, 1),
	PIN_FIELD_BASE(13, 13, 2, 0x080, 0x10, 0, 1),
	PIN_FIELD_BASE(14, 14, 2, 0x080, 0x10, 1, 1),
	PIN_FIELD_BASE(15, 15, 2, 0x080, 0x10, 2, 1),
	PIN_FIELD_BASE(16, 16, 2, 0x080, 0x10, 3, 1),
	PIN_FIELD_BASE(17, 17, 2, 0x080, 0x10, 4, 1),
	PIN_FIELD_BASE(18, 18, 2, 0x080, 0x10, 5, 1),
	PIN_FIELD_BASE(19, 19, 2, 0x080, 0x10, 6, 1),
	PIN_FIELD_BASE(20, 20, 2, 0x080, 0x10, 7, 1),
	PIN_FIELD_BASE(21, 21, 2, 0x080, 0x10, 8, 1),
	PIN_FIELD_BASE(22, 22, 2, 0x080, 0x10, 9, 1),
	PIN_FIELD_BASE(23, 23, 2, 0x080, 0x10, 10, 1),
	PIN_FIELD_BASE(24, 24, 2, 0x080, 0x10, 11, 1),
	PIN_FIELD_BASE(25, 25, 2, 0x080, 0x10, 12, 1),
	PIN_FIELD_BASE(26, 26, 2, 0x080, 0x10, 13, 1),
	PIN_FIELD_BASE(27, 27, 2, 0x080, 0x10, 14, 1),
	PIN_FIELD_BASE(28, 28, 2, 0x080, 0x10, 15, 1),
	PIN_FIELD_BASE(43, 43, 3, 0x080, 0x10, 8, 1),
	PIN_FIELD_BASE(44, 44, 3, 0x080, 0x10, 9, 1),
	PIN_FIELD_BASE(45, 45, 3, 0x080, 0x10, 10, 1),
	PIN_FIELD_BASE(46, 46, 3, 0x080, 0x10, 11, 1),
	PIN_FIELD_BASE(47, 47, 3, 0x080, 0x10, 12, 1),
	PIN_FIELD_BASE(48, 48, 3, 0x080, 0x10, 13, 1),
	PIN_FIELD_BASE(49, 49, 3, 0x080, 0x10, 14, 1),
	PIN_FIELD_BASE(50, 50, 4, 0x080, 0x10, 0, 1),
	PIN_FIELD_BASE(51, 51, 4, 0x080, 0x10, 1, 1),
	PIN_FIELD_BASE(52, 52, 4, 0x080, 0x10, 2, 1),
	PIN_FIELD_BASE(53, 53, 4, 0x080, 0x10, 3, 1),
	PIN_FIELD_BASE(54, 54, 4, 0x080, 0x10, 4, 1),
	PIN_FIELD_BASE(55, 55, 4, 0x080, 0x10, 5, 1),
	PIN_FIELD_BASE(56, 56, 4, 0x080, 0x10, 6, 1),
	PIN_FIELD_BASE(57, 57, 4, 0x080, 0x10, 7, 1),
	PIN_FIELD_BASE(58, 58, 4, 0x080, 0x10, 8, 1),
	PIN_FIELD_BASE(59, 59, 4, 0x080, 0x10, 9, 1),
	PIN_FIELD_BASE(60, 60, 4, 0x080, 0x10, 10, 1),
	PIN_FIELD_BASE(61, 61, 5, 0x080, 0x10, 0, 1),
	PIN_FIELD_BASE(62, 62, 5, 0x080, 0x10, 1, 1),
	PIN_FIELD_BASE(63, 63, 5, 0x080, 0x10, 2, 1),
	PIN_FIELD_BASE(64, 64, 5, 0x080, 0x10, 3, 1),
	PIN_FIELD_BASE(65, 65, 5, 0x080, 0x10, 4, 1),
	PIN_FIELD_BASE(66, 66, 5, 0x080, 0x10, 5, 1),
	PIN_FIELD_BASE(67, 67, 5, 0x080, 0x10, 6, 1),
	PIN_FIELD_BASE(68, 68, 5, 0x080, 0x10, 7, 1),
	PIN_FIELD_BASE(69, 69, 5, 0x080, 0x10, 8, 1),
	PIN_FIELD_BASE(70, 70, 5, 0x080, 0x10, 9, 1),
	PIN_FIELD_BASE(71, 71, 5, 0x080, 0x10, 10, 1),
	PIN_FIELD_BASE(72, 72, 5, 0x080, 0x10, 11, 1),
	PIN_FIELD_BASE(73, 73, 5, 0x080, 0x10, 12, 1),
	PIN_FIELD_BASE(74, 74, 5, 0x080, 0x10, 13, 1),
	PIN_FIELD_BASE(75, 75, 5, 0x080, 0x10, 14, 1),
	PIN_FIELD_BASE(76, 76, 5, 0x080, 0x10, 15, 1),
	PIN_FIELD_BASE(77, 77, 5, 0x080, 0x10, 16, 1),
	PIN_FIELD_BASE(78, 78, 5, 0x080, 0x10, 17, 1),
	PIN_FIELD_BASE(79, 79, 5, 0x080, 0x10, 18, 1),
	PIN_FIELD_BASE(80, 80, 5, 0x080, 0x10, 19, 1),
	PIN_FIELD_BASE(81, 81, 5, 0x080, 0x10, 20, 1),
	PIN_FIELD_BASE(82, 82, 5, 0x080, 0x10, 21, 1),
	PIN_FIELD_BASE(83, 83, 5, 0x080, 0x10, 22, 1),
	PIN_FIELD_BASE(84, 84, 5, 0x080, 0x10, 23, 1),
	PIN_FIELD_BASE(85, 85, 5, 0x080, 0x10, 24, 1),
	PIN_FIELD_BASE(86, 86, 5, 0x080, 0x10, 25, 1),
	PIN_FIELD_BASE(87, 87, 5, 0x080, 0x10, 26, 1),
	PIN_FIELD_BASE(88, 88, 5, 0x080, 0x10, 27, 1),
	PIN_FIELD_BASE(89, 89, 6, 0x080, 0x10, 0, 1),
	PIN_FIELD_BASE(90, 90, 6, 0x080, 0x10, 1, 1),
	PIN_FIELD_BASE(95, 95, 6, 0x080, 0x10, 14, 1),
	PIN_FIELD_BASE(96, 96, 6, 0x080, 0x10, 15, 1),
	PIN_FIELD_BASE(97, 97, 6, 0x080, 0x10, 16, 1),
	PIN_FIELD_BASE(98, 98, 6, 0x080, 0x10, 17, 1),
	PIN_FIELD_BASE(99, 99, 6, 0x080, 0x10, 18, 1),
	PIN_FIELD_BASE(100, 100, 6, 0x080, 0x10, 19, 1),
	PIN_FIELD_BASE(101, 101, 6, 0x080, 0x10, 20, 1),
	PIN_FIELD_BASE(102, 102, 6, 0x080, 0x10, 21, 1),
	PIN_FIELD_BASE(103, 103, 6, 0x080, 0x10, 25, 1),
	PIN_FIELD_BASE(104, 104, 6, 0x080, 0x10, 26, 1),
	PIN_FIELD_BASE(105, 105, 6, 0x080, 0x10, 27, 1),
	PIN_FIELD_BASE(106, 106, 6, 0x080, 0x10, 28, 1),
	PIN_FIELD_BASE(110, 110, 7, 0x080, 0x10, 3, 1),
	PIN_FIELD_BASE(111, 111, 7, 0x080, 0x10, 4, 1),
	PIN_FIELD_BASE(112, 112, 7, 0x080, 0x10, 5, 1),
	PIN_FIELD_BASE(113, 113, 7, 0x080, 0x10, 6, 1),
	PIN_FIELD_BASE(114, 114, 7, 0x080, 0x10, 7, 1),
	PIN_FIELD_BASE(115, 115, 7, 0x080, 0x10, 8, 1),
	PIN_FIELD_BASE(134, 134, 1, 0x080, 0x10, 0, 1),
	PIN_FIELD_BASE(135, 135, 1, 0x080, 0x10, 1, 1),
	PIN_FIELD_BASE(136, 136, 1, 0x080, 0x10, 2, 1),
	PIN_FIELD_BASE(137, 137, 1, 0x080, 0x10, 3, 1),
	PIN_FIELD_BASE(138, 138, 1, 0x080, 0x10, 4, 1),
	PIN_FIELD_BASE(139, 139, 1, 0x080, 0x10, 5, 1),
	PIN_FIELD_BASE(140, 140, 1, 0x080, 0x10, 6, 1),
	PIN_FIELD_BASE(141, 141, 1, 0x080, 0x10, 7, 1),
	PIN_FIELD_BASE(142, 142, 1, 0x080, 0x10, 8, 1),
	PIN_FIELD_BASE(143, 143, 1, 0x080, 0x10, 9, 1),
	PIN_FIELD_BASE(144, 144, 1, 0x080, 0x10, 11, 1),
	PIN_FIELD_BASE(145, 145, 1, 0x080, 0x10, 12, 1),
	PIN_FIELD_BASE(146, 146, 1, 0x080, 0x10, 13, 1),
	PIN_FIELD_BASE(147, 147, 1, 0x080, 0x10, 14, 1),
	PIN_FIELD_BASE(148, 148, 1, 0x080, 0x10, 15, 1),
	PIN_FIELD_BASE(149, 149, 1, 0x080, 0x10, 16, 1),
	PIN_FIELD_BASE(150, 150, 4, 0x080, 0x10, 11, 1),
	PIN_FIELD_BASE(151, 151, 4, 0x080, 0x10, 12, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_pupd_range[] = {
	PIN_FIELD_BASE(29, 29, 2, 0x0C0, 0x10, 2, 1),
	PIN_FIELD_BASE(30, 30, 2, 0x0C0, 0x10, 6, 1),
	PIN_FIELD_BASE(31, 31, 2, 0x0C0, 0x10, 10, 1),
	PIN_FIELD_BASE(32, 32, 2, 0x0C0, 0x10, 14, 1),
	PIN_FIELD_BASE(33, 33, 2, 0x0C0, 0x10, 18, 1),
	PIN_FIELD_BASE(34, 34, 2, 0x0C0, 0x10, 22, 1),
	PIN_FIELD_BASE(35, 35, 3, 0x0C0, 0x10, 2, 1),
	PIN_FIELD_BASE(36, 36, 3, 0x0C0, 0x10, 6, 1),
	PIN_FIELD_BASE(37, 37, 3, 0x0C0, 0x10, 10, 1),
	PIN_FIELD_BASE(38, 38, 3, 0x0C0, 0x10, 14, 1),
	PIN_FIELD_BASE(39, 39, 3, 0x0C0, 0x10, 18, 1),
	PIN_FIELD_BASE(40, 40, 3, 0x0C0, 0x10, 22, 1),
	PIN_FIELD_BASE(41, 41, 3, 0x0C0, 0x10, 26, 1),
	PIN_FIELD_BASE(42, 42, 3, 0x0C0, 0x10, 30, 1),
	PIN_FIELD_BASE(91, 91, 6, 0x0C0, 0x10, 6, 1),
	PIN_FIELD_BASE(92, 92, 6, 0x0C0, 0x10, 2, 1),
	PIN_FIELD_BASE(93, 93, 6, 0x0C0, 0x10, 10, 1),
	PIN_FIELD_BASE(94, 94, 6, 0x0C0, 0x10, 14, 1),
	PIN_FIELD_BASE(107, 107, 7, 0x0C0, 0x10, 2, 1),
	PIN_FIELD_BASE(108, 108, 7, 0x0C0, 0x10, 6, 1),
	PIN_FIELD_BASE(109, 109, 7, 0x0C0, 0x10, 10, 1),
	PIN_FIELD_BASE(116, 116, 7, 0x0C0, 0x10, 14, 1),
	PIN_FIELD_BASE(117, 117, 7, 0x0C0, 0x10, 18, 1),
	PIN_FIELD_BASE(118, 118, 7, 0x0C0, 0x10, 22, 1),
	PIN_FIELD_BASE(119, 119, 7, 0x0C0, 0x10, 26, 1),
	PIN_FIELD_BASE(120, 120, 7, 0x0C0, 0x10, 30, 1),
	PIN_FIELD_BASE(121, 121, 7, 0x0D0, 0x10, 2, 1),
	PIN_FIELD_BASE(122, 122, 8, 0x0C0, 0x10, 2, 1),
	PIN_FIELD_BASE(123, 123, 8, 0x0C0, 0x10, 6, 1),
	PIN_FIELD_BASE(124, 124, 8, 0x0C0, 0x10, 10, 1),
	PIN_FIELD_BASE(125, 125, 8, 0x0C0, 0x10, 14, 1),
	PIN_FIELD_BASE(126, 126, 8, 0x0C0, 0x10, 18, 1),
	PIN_FIELD_BASE(127, 127, 8, 0x0C0, 0x10, 22, 1),
	PIN_FIELD_BASE(128, 128, 8, 0x0C0, 0x10, 26, 1),
	PIN_FIELD_BASE(129, 129, 8, 0x0C0, 0x10, 30, 1),
	PIN_FIELD_BASE(130, 130, 8, 0x0D0, 0x10, 2, 1),
	PIN_FIELD_BASE(131, 131, 8, 0x0D0, 0x10, 6, 1),
	PIN_FIELD_BASE(132, 132, 8, 0x0D0, 0x10, 10, 1),
	PIN_FIELD_BASE(133, 133, 8, 0x0D0, 0x10, 14, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_r1_range[] = {
	PIN_FIELD_BASE(29, 29, 2, 0x0C0, 0x10, 1, 1),
	PIN_FIELD_BASE(30, 30, 2, 0x0C0, 0x10, 5, 1),
	PIN_FIELD_BASE(31, 31, 2, 0x0C0, 0x10, 9, 1),
	PIN_FIELD_BASE(32, 32, 2, 0x0C0, 0x10, 13, 1),
	PIN_FIELD_BASE(33, 33, 2, 0x0C0, 0x10, 17, 1),
	PIN_FIELD_BASE(34, 34, 2, 0x0C0, 0x10, 21, 1),
	PIN_FIELD_BASE(35, 35, 3, 0x0C0, 0x10, 1, 1),
	PIN_FIELD_BASE(36, 36, 3, 0x0C0, 0x10, 5, 1),
	PIN_FIELD_BASE(37, 37, 3, 0x0C0, 0x10, 9, 1),
	PIN_FIELD_BASE(38, 38, 3, 0x0C0, 0x10, 13, 1),
	PIN_FIELD_BASE(39, 39, 3, 0x0C0, 0x10, 17, 1),
	PIN_FIELD_BASE(40, 40, 3, 0x0C0, 0x10, 21, 1),
	PIN_FIELD_BASE(41, 41, 3, 0x0C0, 0x10, 25, 1),
	PIN_FIELD_BASE(42, 42, 3, 0x0C0, 0x10, 29, 1),
	PIN_FIELD_BASE(91, 91, 6, 0x0C0, 0x10, 5, 1),
	PIN_FIELD_BASE(92, 92, 6, 0x0C0, 0x10, 1, 1),
	PIN_FIELD_BASE(93, 93, 6, 0x0C0, 0x10, 9, 1),
	PIN_FIELD_BASE(94, 94, 6, 0x0C0, 0x10, 13, 1),
	PIN_FIELD_BASE(107, 107, 7, 0x0C0, 0x10, 1, 1),
	PIN_FIELD_BASE(108, 108, 7, 0x0C0, 0x10, 5, 1),
	PIN_FIELD_BASE(109, 109, 7, 0x0C0, 0x10, 9, 1),
	PIN_FIELD_BASE(116, 116, 7, 0x0C0, 0x10, 13, 1),
	PIN_FIELD_BASE(117, 117, 7, 0x0C0, 0x10, 17, 1),
	PIN_FIELD_BASE(118, 118, 7, 0x0C0, 0x10, 21, 1),
	PIN_FIELD_BASE(119, 119, 7, 0x0C0, 0x10, 25, 1),
	PIN_FIELD_BASE(120, 120, 7, 0x0C0, 0x10, 29, 1),
	PIN_FIELD_BASE(121, 121, 7, 0x0D0, 0x10, 1, 1),
	PIN_FIELD_BASE(122, 122, 8, 0x0C0, 0x10, 1, 1),
	PIN_FIELD_BASE(123, 123, 8, 0x0C0, 0x10, 5, 1),
	PIN_FIELD_BASE(124, 124, 8, 0x0C0, 0x10, 9, 1),
	PIN_FIELD_BASE(125, 125, 8, 0x0C0, 0x10, 13, 1),
	PIN_FIELD_BASE(126, 126, 8, 0x0C0, 0x10, 17, 1),
	PIN_FIELD_BASE(127, 127, 8, 0x0C0, 0x10, 21, 1),
	PIN_FIELD_BASE(128, 128, 8, 0x0C0, 0x10, 25, 1),
	PIN_FIELD_BASE(129, 129, 8, 0x0C0, 0x10, 29, 1),
	PIN_FIELD_BASE(130, 130, 8, 0x0D0, 0x10, 1, 1),
	PIN_FIELD_BASE(131, 131, 8, 0x0D0, 0x10, 5, 1),
	PIN_FIELD_BASE(132, 132, 8, 0x0D0, 0x10, 9, 1),
	PIN_FIELD_BASE(133, 133, 8, 0x0D0, 0x10, 13, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_r0_range[] = {
	PIN_FIELD_BASE(29, 29, 2, 0x0C0, 0x10, 0, 1),
	PIN_FIELD_BASE(30, 30, 2, 0x0C0, 0x10, 4, 1),
	PIN_FIELD_BASE(31, 31, 2, 0x0C0, 0x10, 8, 1),
	PIN_FIELD_BASE(32, 32, 2, 0x0C0, 0x10, 12, 1),
	PIN_FIELD_BASE(33, 33, 2, 0x0C0, 0x10, 16, 1),
	PIN_FIELD_BASE(34, 34, 2, 0x0C0, 0x10, 20, 1),
	PIN_FIELD_BASE(35, 35, 3, 0x0C0, 0x10, 0, 1),
	PIN_FIELD_BASE(36, 36, 3, 0x0C0, 0x10, 4, 1),
	PIN_FIELD_BASE(37, 37, 3, 0x0C0, 0x10, 8, 1),
	PIN_FIELD_BASE(38, 38, 3, 0x0C0, 0x10, 12, 1),
	PIN_FIELD_BASE(39, 39, 3, 0x0C0, 0x10, 16, 1),
	PIN_FIELD_BASE(40, 40, 3, 0x0C0, 0x10, 20, 1),
	PIN_FIELD_BASE(41, 41, 3, 0x0C0, 0x10, 24, 1),
	PIN_FIELD_BASE(42, 42, 3, 0x0C0, 0x10, 28, 1),
	PIN_FIELD_BASE(91, 91, 6, 0x0C0, 0x10, 4, 1),
	PIN_FIELD_BASE(92, 92, 6, 0x0C0, 0x10, 0, 1),
	PIN_FIELD_BASE(93, 93, 6, 0x0C0, 0x10, 8, 1),
	PIN_FIELD_BASE(94, 94, 6, 0x0C0, 0x10, 12, 1),
	PIN_FIELD_BASE(107, 107, 7, 0x0C0, 0x10, 0, 1),
	PIN_FIELD_BASE(108, 108, 7, 0x0C0, 0x10, 4, 1),
	PIN_FIELD_BASE(109, 109, 7, 0x0C0, 0x10, 8, 1),
	PIN_FIELD_BASE(116, 116, 7, 0x0C0, 0x10, 12, 1),
	PIN_FIELD_BASE(117, 117, 7, 0x0C0, 0x10, 16, 1),
	PIN_FIELD_BASE(118, 118, 7, 0x0C0, 0x10, 20, 1),
	PIN_FIELD_BASE(119, 119, 7, 0x0C0, 0x10, 24, 1),
	PIN_FIELD_BASE(120, 120, 7, 0x0C0, 0x10, 28, 1),
	PIN_FIELD_BASE(121, 121, 7, 0x0D0, 0x10, 0, 1),
	PIN_FIELD_BASE(122, 122, 8, 0x0C0, 0x10, 0, 1),
	PIN_FIELD_BASE(123, 123, 8, 0x0C0, 0x10, 4, 1),
	PIN_FIELD_BASE(124, 124, 8, 0x0C0, 0x10, 8, 1),
	PIN_FIELD_BASE(125, 125, 8, 0x0C0, 0x10, 12, 1),
	PIN_FIELD_BASE(126, 126, 8, 0x0C0, 0x10, 16, 1),
	PIN_FIELD_BASE(127, 127, 8, 0x0C0, 0x10, 20, 1),
	PIN_FIELD_BASE(128, 128, 8, 0x0C0, 0x10, 24, 1),
	PIN_FIELD_BASE(129, 129, 8, 0x0C0, 0x10, 28, 1),
	PIN_FIELD_BASE(130, 130, 8, 0x0D0, 0x10, 0, 1),
	PIN_FIELD_BASE(131, 131, 8, 0x0D0, 0x10, 4, 1),
	PIN_FIELD_BASE(132, 132, 8, 0x0D0, 0x10, 8, 1),
	PIN_FIELD_BASE(133, 133, 8, 0x0D0, 0x10, 12, 1),
};

static const struct mtk_pin_field_calc mt6763_pin_drv_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(1, 1, 6, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(2, 2, 6, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(3, 3, 6, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(4, 4, 6, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(5, 5, 6, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(6, 6, 6, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(7, 7, 6, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(8, 8, 6, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(9, 9, 6, 0x0B0, 0x10, 12, 4),
	PIN_FIELD_BASE(10, 10, 6, 0x0B0, 0x10, 12, 4),
	PIN_FIELD_BASE(11, 11, 1, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(12, 12, 1, 0x0A0, 0x10, 28, 4),
	PIN_FIELD_BASE(13, 13, 2, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(14, 14, 2, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(15, 15, 2, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(16, 16, 2, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(17, 17, 2, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(18, 18, 2, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(19, 19, 2, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(20, 20, 2, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(21, 21, 2, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(22, 22, 2, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(23, 23, 2, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(24, 24, 2, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(25, 25, 2, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(26, 26, 2, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(27, 27, 2, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(28, 28, 2, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(29, 29, 2, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(30, 30, 2, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(31, 31, 2, 0x0A0, 0x10, 24, 4),
	PIN_FIELD_BASE(32, 32, 2, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(33, 33, 2, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(34, 34, 2, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(35, 35, 3, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(36, 36, 3, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(37, 37, 3, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(38, 38, 3, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(39, 39, 3, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(40, 40, 3, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(41, 41, 3, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(42, 42, 3, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(43, 43, 3, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(44, 44, 3, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(45, 45, 3, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(46, 46, 3, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(47, 47, 3, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(48, 48, 3, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(49, 49, 3, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(50, 50, 4, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(51, 51, 4, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(52, 52, 4, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(53, 53, 4, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(54, 54, 4, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(55, 55, 4, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(56, 56, 4, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(57, 57, 4, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(58, 58, 4, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(59, 59, 4, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(60, 60, 4, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(61, 61, 5, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(62, 62, 5, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(63, 63, 5, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(64, 64, 5, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(65, 65, 5, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(66, 66, 5, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(67, 67, 5, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(68, 68, 5, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(69, 69, 5, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(70, 70, 5, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(71, 71, 5, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(72, 72, 5, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(73, 73, 5, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(74, 74, 5, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(75, 75, 5, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(76, 76, 5, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(77, 77, 5, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(78, 78, 5, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(79, 79, 5, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(80, 80, 5, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(81, 81, 5, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(82, 82, 5, 0x0A0, 0x10, 24, 4),
	PIN_FIELD_BASE(83, 83, 5, 0x0A0, 0x10, 24, 4),
	PIN_FIELD_BASE(84, 84, 5, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(85, 85, 5, 0x0A0, 0x10, 28, 4),
	PIN_FIELD_BASE(86, 86, 5, 0x0A0, 0x10, 28, 4),
	PIN_FIELD_BASE(87, 87, 5, 0x0A0, 0x10, 28, 4),
	PIN_FIELD_BASE(88, 88, 5, 0x0A0, 0x10, 28, 4),
	PIN_FIELD_BASE(89, 89, 6, 0x0B0, 0x10, 8, 4),
	PIN_FIELD_BASE(90, 90, 6, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(91, 91, 6, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(92, 92, 6, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(93, 93, 6, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(94, 94, 6, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(95, 95, 6, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(96, 96, 6, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(97, 97, 6, 0x0A0, 0x10, 24, 4),
	PIN_FIELD_BASE(98, 98, 6, 0x0A0, 0x10, 24, 4),
	PIN_FIELD_BASE(99, 99, 6, 0x0A0, 0x10, 28, 4),
	PIN_FIELD_BASE(100, 100, 6, 0x0B0, 0x10, 0, 4),
	PIN_FIELD_BASE(101, 101, 6, 0x0B0, 0x10, 4, 4),
	PIN_FIELD_BASE(102, 102, 6, 0x0B0, 0x10, 4, 4),
	PIN_FIELD_BASE(103, 103, 6, 0x0B0, 0x10, 16, 4),
	PIN_FIELD_BASE(104, 104, 6, 0x0B0, 0x10, 16, 4),
	PIN_FIELD_BASE(105, 105, 6, 0x0B0, 0x10, 20, 4),
	PIN_FIELD_BASE(106, 106, 6, 0x0B0, 0x10, 20, 4),
	PIN_FIELD_BASE(107, 107, 7, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(108, 108, 7, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(109, 109, 7, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(110, 110, 7, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(111, 111, 7, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(112, 112, 7, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(113, 113, 7, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(114, 114, 7, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(115, 115, 7, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(116, 116, 7, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(117, 117, 7, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(118, 118, 7, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(119, 119, 7, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(120, 120, 7, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(121, 121, 7, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(122, 122, 8, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(123, 123, 8, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(124, 124, 8, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(125, 125, 8, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(126, 126, 8, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(127, 127, 8, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(128, 128, 8, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(129, 129, 8, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(130, 130, 8, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(131, 131, 8, 0x0A0, 0x10, 12, 4),
	PIN_FIELD_BASE(132, 132, 8, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(133, 133, 8, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(134, 134, 1, 0x0A0, 0x10, 0, 4),
	PIN_FIELD_BASE(135, 135, 1, 0x0A0, 0x10, 4, 4),
	PIN_FIELD_BASE(136, 136, 1, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(137, 137, 1, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(138, 138, 1, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(139, 139, 1, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(140, 140, 1, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(141, 141, 1, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(142, 142, 1, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(143, 143, 1, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(144, 144, 1, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(145, 145, 1, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(146, 146, 1, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(147, 147, 1, 0x0A0, 0x10, 16, 4),
	PIN_FIELD_BASE(148, 148, 1, 0x0A0, 0x10, 20, 4),
	PIN_FIELD_BASE(149, 149, 1, 0x0A0, 0x10, 24, 4),
	PIN_FIELD_BASE(150, 150, 4, 0x0A0, 0x10, 8, 4),
	PIN_FIELD_BASE(151, 151, 4, 0x0A0, 0x10, 8, 4),
};

static const struct mtk_pin_reg_calc mt6763_reg_cals[PINCTRL_PIN_REG_MAX] = {
	[PINCTRL_PIN_REG_MODE] = MTK_RANGE(mt6763_pin_mode_range),
	[PINCTRL_PIN_REG_DIR] = MTK_RANGE(mt6763_pin_dir_range),
	[PINCTRL_PIN_REG_DI] = MTK_RANGE(mt6763_pin_di_range),
	[PINCTRL_PIN_REG_DO] = MTK_RANGE(mt6763_pin_do_range),
	[PINCTRL_PIN_REG_SR] = MTK_RANGE(mt6763_pin_dir_range),
	[PINCTRL_PIN_REG_SMT] = MTK_RANGE(mt6763_pin_smt_range),
	[PINCTRL_PIN_REG_IES] = MTK_RANGE(mt6763_pin_ies_range),
	[PINCTRL_PIN_REG_PULLEN] = MTK_RANGE(mt6763_pin_pullen_range),
	[PINCTRL_PIN_REG_PULLSEL] = MTK_RANGE(mt6763_pin_pullsel_range),
	[PINCTRL_PIN_REG_DRV] = MTK_RANGE(mt6763_pin_drv_range),
	[PINCTRL_PIN_REG_PUPD] = MTK_RANGE(mt6763_pin_pupd_range),
	[PINCTRL_PIN_REG_R0] = MTK_RANGE(mt6763_pin_r0_range),
	[PINCTRL_PIN_REG_R1] = MTK_RANGE(mt6763_pin_r1_range),
};

static const struct mtk_eint_hw mt6763_eint_hw = {
	.port_mask = 7,
	.ports     = 6,
	.ap_num    = 212,
	.db_cnt    = 13,
};

static const struct mtk_pin_soc mt6763_data = {
	.reg_cal = mt6763_reg_cals,
	.pins = mtk_pins_mt6763,
	.npins = ARRAY_SIZE(mtk_pins_mt6763),
	.ngrps = ARRAY_SIZE(mtk_pins_mt6763),
	.nfuncs = 8,
	.eint_hw = &mt6763_eint_hw,
	.gpio_m = 0,
	.bias_set_combo = mtk_pinconf_bias_set_combo,
	.bias_get_combo = mtk_pinconf_bias_get_combo,
	.drive_set = mtk_pinconf_drive_set_direct_val,
	.drive_get = mtk_pinconf_drive_get_direct_val,
};

static const struct of_device_id mt6763_pinctrl_of_match[] = {
	{ .compatible = "mediatek,mt6763-pinctrl", },
	{ }
};

static int mt6763_pinctrl_probe(struct platform_device *pdev)
{
	return mtk_paris_pinctrl_probe(pdev, &mt6763_data);
}

static struct platform_driver mt6763_pinctrl_driver = {
	.driver = {
		.name = "mt6763-pinctrl",
		.of_match_table = mt6763_pinctrl_of_match,
	},
	.probe = mt6763_pinctrl_probe,
};

static int __init mt6763_pinctrl_init(void)
{
	return platform_driver_register(&mt6763_pinctrl_driver);
}
arch_initcall(mt6763_pinctrl_init);
